/*
 Get acceleration data from Texas Instruments Chronos watch (eZ430).
 Move mouse using X & Y accelleration.
 
 Original code by Sean Brewer and Oliver Smith
 http://chemicaloliver.net/programming/graphing-ti-ez430-chronos-watch-data-in-processing/
 http://e2e.ti.com/support/microcontrollers/msp43016-bit_ultra-low_power_mcus/f/166/t/32714.aspx

 Excellent Kalman information found at Marcus's site
 http://interactive-matter.org/2009/12/filtering-sensor-data-with-a-kalman-filter/
 
 Additions by Michel Blanc.
 Copyright (c) 2010 Sean Brewer / Oliver Smith / Michel Blanc / ERASME
 
 Permission is hereby granted, free of charge, to any person
 obtaining a copy of this software and associated documentation
 files (the "Software"), to deal in the Software without
 restriction, including without limitation the rights to use,
 copy, modify, merge, publish, distribute, sublicense, and/or sell
 copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following
 conditions:

 The above copyright notice and this permission notice shall be
 included in all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 OTHER DEALINGS IN THE SOFTWARE.

If you want you may contact me at mblanc@erasme.org
or on jabber: leucos@jabber.org
*/

import java.awt.*;
import java.awt.event.InputEvent;

// Kalman objects for X, Y axis
Kalman kx;
Kalman ky;

// Watch object
Ez430 watch;

// Initial Kalman values
float q = 0.0625;
float r = 32.0;
float p = 1.3833;

Robot robot;

void setup() {
  //Draw window
  size(400, 400);
  background(0);

  smooth();

  watch = new Ez430(this, "/dev/ttyACM0");
  watch.debug(false);
  
  initKalman();
  blankScreen();

  try { 
      robot = new Robot();
  } 
  catch (AWTException e) {
    e.printStackTrace(); 
  }
} 

void blankScreen() {
  background(0);
  stroke(255,255,255);
  strokeWeight(2);
  noFill();
  
  for (int i=0; i<5; i++) {
    ellipse(width/2, height/2, i*25+25, i*25+25);
  }

  line(50, height/2, width-50, height/2);
  line(width/2, 50, width/2, height-50);  
}

void initKalman() {
  print("Waiting for watch...");
  while(watch.Available() == 0) {};
  print("ok\nReading initial values...");
  watch.ReadValues();
  print ("ok\nInitializing kalman filters...");
  
  kx = new Kalman (q, r, p, watch.x());
  ky = new Kalman (q, r, p, watch.y());

  println("ok");
}

void draw() {
  if ((watch.Available() > 0) && watch.ReadValues()) {
    blankScreen();

    // update Kalman for x
    kx.Update(watch.x());
      
    // update Kalman for y
    ky.Update(watch.y());
      
    // We add a 1.5 factor since values close to 128 are never achieved
    int deltaX = (int)map(ky.Value()* 1.5, -128, 128, 0, width);
    int deltaY =  (int)map(kx.Value() * 1.5, -128, 128, 0, height);
      
    int deltaMouseX = (int)map(ky.Value(), -128, 128, -100, 100);
    int deltaMouseY =  (int)map(kx.Value(), -128, 128, -100, 100);
      
    int mX = MouseInfo.getPointerInfo().getLocation().x + deltaMouseX;
    int mY = MouseInfo.getPointerInfo().getLocation().y + deltaMouseY;
  
    strokeWeight(1);  
    fill(230,230,230);
    ellipse(deltaX, deltaY, 10, 10);
      
    robot.mouseMove(mX, mY);
      
    if (watch.button(Ez430.TOPLEFT)) {
        println("Pressing LMB");        
        robot.mousePress(InputEvent.BUTTON1_MASK);
        robot.mouseRelease(InputEvent.BUTTON1_MASK);          
    } else if (watch.button(Ez430.TOPRIGHT)) {
        println("Pressing RMB");
        robot.mousePress(InputEvent.BUTTON3_MASK);
        robot.mouseRelease(InputEvent.BUTTON3_MASK);          
    }
  }    
}

