/*
 Texas Instruments Chronos watch (eZ430) Class. 

 Original code by Sean Brewer and Oliver Smith
 http://chemicaloliver.net/programming/graphing-ti-ez430-chronos-watch-data-in-processing/
 http://e2e.ti.com/support/microcontrollers/msp43016-bit_ultra-low_power_mcus/f/166/t/32714.aspx
 
 Copyright (c) 2010 Sean Brewer / Oliver Smith / Michel Blanc
 
 Permission is hereby granted, free of charge, to any person
 obtaining a copy of this software and associated documentation
 files (the "Software"), to deal in the Software without
 restriction, including without limitation the rights to use,
 copy, modify, merge, publish, distribute, sublicense, and/or sell
 copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following
 conditions:

 The above copyright notice and this permission notice shall be
 included in all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 OTHER DEALINGS IN THE SOFTWARE.

If you want you may contact me at mblanc@erasme.org
or on jabber: leucos@jabber.org
*/

import processing.serial.*;

public class Ez430 {
  static final int TOPLEFT = 0;
  static final int TOPRIGHT = 1;
  static final int BOTTOMLEFT = 2;  
  
  private Serial chronos;
  private boolean debug;
  
  private float[] axis = new float[3];
  private boolean[] buttonState = new boolean[3];
    
  //Take what we know about the packets for starting the access point
  //and put it in its integer representation
  private int startAccessPointNum[] = {255, 7, 3};

  //Take what we know about the packets for aquiring data from the chronos
  //and put it in its integer representation
  private int accDataRequestNum[] = {255, 8, 7, 0, 0, 0, 0};

  //Convert it all to bytes so that watch will understand
  //what we're talking about..
  private byte startAccessPoint[] = byte(startAccessPointNum);
  private byte accDataRequest[] = byte(accDataRequestNum);
  
  public Ez430(PApplet applet, String port) {
    //In linux /dev/ttyACM0 is (sometimes) not correctly detected by java so 
    //setting this property solves that
    //You'll need to change this if you are using osx/windows or 
    //another serial port
    System.setProperty("gnu.io.rxtx.SerialPorts", port);
    chronos = new Serial(applet, port, 115200);
    
    //Start the access point..
    chronos.write(startAccessPoint);

    //Until the port is still availible...
    //Send data request to chronos.
    chronos.write(accDataRequest);
  }
  
  public int Available() {
    chronos.write(accDataRequest);    
    return chronos.available();
  }
  
  public boolean ReadValues() {
    //Accelerometer data is 7 bytes long
    int[] buf = new int[7];  
    
    for (int i = 0; i < 7; i++)
      buf[i] = chronos.read();

    //Fourth byte indicates if there are coordinates. If the byte is 0xFF (255)
    //Then there is no data. If it is 1, then data is valid.
    //If byte is 17, 49 or 33, it's a button.

   /* for (int i = 0; i < 7; i++)
      print (buf[i]+" ");
    println();
    */
  
    if (buf[3] == 1)  {
      for (int j=0; j < 3; j++) {
        axis[j] = parseFloat(str(buf[j+4]));
        // Map between [-128,128]
        if (axis[j] > 128) 
          axis[j] -=  255;
      }

      debugPrintln("x: " + str(axis[0]) + " y: " + str(axis[1]) + " z: " + str(axis[2]));
      return true;
    } else if(buf[3] == 17) {
      debugPrintln("button : top left");
      buttonState[TOPLEFT] = true;
      return true;
    } else if (buf[3] == 49) {
      debugPrintln("button : top right");
      buttonState[TOPRIGHT] = true;        
      return true;
    } else if (buf[3] == 33) {
      debugPrintln("button : bottom left");
      buttonState[BOTTOMLEFT] = true;            
      return true;
    }
    resetButtons();
    return false;
  }
  
  private void resetButtons() {
    buttonState[TOPLEFT] = buttonState[TOPRIGHT] = buttonState[BOTTOMLEFT] = false;    
  }
  
  public boolean[] button() {
    return buttonState;
  }

  public boolean button(int buttonId) {
    if ((buttonId >= 0) && (buttonId < buttonState.length)) {
      if (buttonState[buttonId]) 
        print("#");
      return buttonState[buttonId];
    } else {
      return false;
    }
  }
    
  public float x() {
    return axis[0];
  }
  
  public float y() {
    return axis[1];
  }

  public float z() {
    return axis[2];
  }
  
  public void debug(boolean state) {
    debug = state;
  }
  
  public boolean debug() {
    return debug;
  }
  
  public void debugPrintln(String thestr) {
    if (debug) 
      println(thestr);
  }
  
  public void debugPrint(String thestr) {
    if (debug) 
      print(thestr);
  }  
}
