/*
 Get acceleration data from Texas Instruments Chronos watch (eZ430).
 Send it via OSC
 
 Original watch code by Sean Brewer and Oliver Smith
 http://chemicaloliver.net/programming/graphing-ti-ez430-chronos-watch-data-in-processing/
 http://e2e.ti.com/support/microcontrollers/msp43016-bit_ultra-low_power_mcus/f/166/t/32714.aspx

 Excellent Kalman information found at Marcus's site
 http://interactive-matter.org/2009/12/filtering-sensor-data-with-a-kalman-filter/
 
 Additions by Michel Blanc.
 Copyright (c) 2010 Sean Brewer / Oliver Smith / Michel Blanc / ERASME
 
 Permission is hereby granted, free of charge, to any person
 obtaining a copy of this software and associated documentation
 files (the "Software"), to deal in the Software without
 restriction, including without limitation the rights to use,
 copy, modify, merge, publish, distribute, sublicense, and/or sell
 copies of the Software, and to permit persons to whom the
 Software is furnished to do so, subject to the following
 conditions:

 The above copyright notice and this permission notice shall be
 included in all copies or substantial portions of the Software.

 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 OTHER DEALINGS IN THE SOFTWARE.

If you want you may contact me at mblanc@erasme.org
or on jabber: leucos@jabber.org
*/

import oscP5.*;
import netP5.*;

// Kalman objects for X, Y axis
Kalman kx;
Kalman ky;
Kalman kz;

// Watch object
Ez430 watch;

// Initial Kalman values
float q = 0.0625;
float r = 32.0;
float p = 1.3833;

// Frame number of last sent message
int lastSent = 0;

OscP5 oscP5;
NetAddress myRemoteLocation;

void setup() {
  //Draw window
  size(400, 400);
  background(0);

  smooth();

  watch = new Ez430(this, "/dev/ttyACM0");
  watch.debug(true);
  
  oscP5 = new OscP5(this,12000);
  myRemoteLocation = new NetAddress("255.255.255.255",1234);
  
  initKalman();
} 

void initKalman() {
  print("Waiting for watch...");
  while(watch.Available() == 0) {};
  print("ok\nReading initial values...");
  watch.ReadValues();
  print ("ok\nInitializing kalman filters...");
  
  kx = new Kalman (q, r, p, watch.x());
  ky = new Kalman (q, r, p, watch.y());
  kz = new Kalman (q, r, p, watch.z());
  
  println("ok");
}

void draw() {  
  background(0);
  
  if (watch.Available() > 0 && watch.ReadValues()) {
    // update Kalman for x
    kx.Update(watch.x());
      
    // update Kalman for y
    ky.Update(watch.y());
      
    // update Kalman for z
    kz.Update(watch.z());

    /* create a new OscMessage with an address pattern, in this case /test. */
    OscMessage myOscMessage = new OscMessage("/erasme/ez430");
    
    /* add a value (an integer) to the OscMessage */
    myOscMessage.add(kx.Value());
    myOscMessage.add(ky.Value());
    myOscMessage.add(kz.Value());
    myOscMessage.add(watch.button(Ez430.TOPLEFT));
    myOscMessage.add(watch.button(Ez430.TOPRIGHT));
    myOscMessage.add(watch.button(Ez430.BOTTOMLEFT));
    
    /* send the OscMessage to a remote location specified in myNetAddress */
    oscP5.send(myOscMessage, myRemoteLocation);
    lastSent = frameCount;
    
    print (".");
  }
  
  if (frameCount < (lastSent + 255)) {
    fill(230,30,30, 255 - (frameCount - lastSent));
    ellipse(width/2, height/2, 30, 30);
  }
} 

